{This unit provides a generic listbox control and buttons, whose
events can be assigned to procedures stored within the body of THIS
unit, not the calling unit.  Other units can assign these internal
procedures to the available buttons, events, etc. programmatically.

The event-handling procedures contained in this unit could certainly
be placed inside the calling unit for good organizational practices.
They are placed in this unit simply to increase clarity of the
NWLib demo program by reducing interface code. }

unit List;

interface

uses
    sysUtils,
    WinTypes,
    WinProcs,
    Classes,
    Graphics,
    Forms,
    Controls,
    Buttons,
    StdCtrls,
    ExtCtrls,
    Grids,
    dialogs,
    Nwtools,
    Nwlib,
    NwProp, 
    NWprint;

type
  TwinList = class(TForm)
    Panel1: TPanel;
    buttonPanel: TPanel;
    listGrid: TStringGrid;
    Button1: TButton;
    Button2: TButton;
    Button3: TButton;
    Button4: TButton;
    NWLib1: TNWLib;
    NWTools1: TNWTools;
    NWProp1: TNWProp;
    NWPrint1: TNWPrint;
    procedure FormPaint(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    inObjectName : string ;
    inServer     : TNWConnHandle ;
    inQueue      : TObjID ;
    procedure onPrintJobsShow(sender : TObject) ;
    procedure onPropertiesShow(sender : TObject)  ;
    procedure onGroupShow(sender : TObject)  ;
    procedure onRightsShow(sender : TObject)  ;
    procedure onMemberShow(sender : TObject)  ;
    procedure UserAddGroup(sender : TObject) ;
    procedure groupAddUser(sender : TObject) ;
    procedure userDelFromGroup(sender  : TObject) ;
    procedure groupDelUser(sender  : TObject) ;
    procedure objSeeObjectInfo(sender  : TObject) ;
    procedure showPropertyInfo(sender  : TObject) ;
    procedure createNewProperty(sender : TObject) ;
    procedure deleteObjProperty(sender : TObject) ;
    procedure addNewRight(sender : TObject) ;
    procedure deleteRight(sender : TObject) ;
    procedure editRight(sender : TObject) ;
    procedure queueJobInfo(sender : TObject) ;
    procedure queueJobDelete(sender : TObject) ;
  end;

var
  winList: TwinList;

implementation

{$R *.DFM}

uses
  trustee ;

procedure TwinList.FormShow(Sender: TObject);
  {set grid's lineheight to current font (nwtools.pas)}
  begin
    autoGridLineHeight(listGrid) ;
  end;

procedure TwinList.FormPaint(Sender: TObject);
  {enable/disable buttons}
  begin
    button1.enabled := (listGrid.rowCount > 0) ;
    button2.enabled :=  button1.enabled ;
    button3.enabled :=  button1.enabled ;
  end;

{********************* formShow methods **********************}

procedure TWinList.onPrintJobsShow(sender : TObject) ;
  { used when a print queue is double-clicked to show jobs}
  var
    ncursor  : TCursor ;
    tempList : TStringList ;
    nLoop    : word ;
  begin
    ncursor       := screen.cursor ;
    screen.cursor := crHourglass ;
    tempList      := TStringList.create ;
    caption       := 'Print Queue Contents' ;
    { Edit listGrid's Button Properties }
    { Loads onClick events, edits captions }
    button1.caption := '&Delete'   ;
    button2.caption := '&Info'     ;
    button3.caption := '&Refresh'  ;
    button4.caption := '&Quit'     ;
    button1.onClick := queueJobDelete ;
    button2.onClick := queueJobInfo ;
    button3.onClick := onPrintJobsShow ;
    button4.modalResult := mrCancel ;

    {get queue jobs and fill up the grid}
    if getQueueJobList(inServer,
                       getObjName(inServer,inQueue),
                       tempList) then
      begin
        listGrid.rowCount    := (tempList.count+1) ;
        listGrid.cells[0,0]  := 'Owner        Status     Description'  ;
        listGrid.onDblClick  := button3.onClick    ;
        for nLoop := 1 to tempList.count do begin
          listGrid.cells[0,nloop] := tempList[nloop-1] ;
          listGrid.cells[1,nloop] := intToStr(TNWQueueJobID(tempList.objects[nloop-1])) ;
        end;
        if (listGrid.rowCount > 1) then
          begin
            listGrid.fixedRows := 1 ;
            listGrid.row := 1 ;
          end;
      end;
    tempList.free ;
    screen.cursor := ncursor ;
  end;


procedure TWinList.onGroupShow(sender : TObject)  ;
  { used when objEdit's 'My Groups' button is selected }
  var
    ncursor  : TCursor ;
    tempList : TStringList ;
    nLoop    : word ;
  begin
    ncursor       := screen.cursor ;
    screen.cursor := crHourglass ;
    tempList      := TStringList.create ;
    caption       := inObjectName + ': Groups I''m In' ;
    { Edit listGrid's Button Properties }
    { Loads onClick events, edits captions }
    button1.caption := '&Add'     ;
    button2.caption := '&Delete'  ;
    button3.caption := '&Info'    ;
    button4.caption := '&Quit'    ;
    button1.onClick := userAddGroup     ;
    button2.onClick := userDelFromGroup ;
    button3.onClick := objSeeObjectInfo ;
    button4.modalResult := mrCancel ;
    { Edit and Fill Up the listGrid }
    tempList := getMyGroups(inServer,
                            inObjectName) ;
    listGrid.rowCount     := (tempList.count+1) ;
    listGrid.cells[0,0]   := 'Group Name'       ;
    listGrid.onDblClick   := button3.onClick    ;
    for nLoop := 1 to tempList.count do
      listGrid.cells[0,nloop] := tempList[nloop-1] ;
    if (listGrid.rowCount > 1) then
      begin
        listGrid.fixedRows := 1 ;
        listGrid.row := 1 ;
      end;
    tempList.free ;
    screen.cursor := ncursor ;
  end;

procedure TWinList.onMemberShow(sender : TObject)  ;
  { used when objEdit's or winList's 'Members' button is selected }
  var
    ncursor  : TCursor ;
    tempList : TStringList ;
    nLoop    : word ;
  begin
    ncursor       := screen.cursor ;
    screen.cursor := crHourglass ;
    tempList      := TStringList.create ;
    caption       := inObjectName + ': Group Members' ;
    { Edit listGrid's Button Properties }
    { Loads onClick events, edits captions }
    button1.caption := '&Add'     ;
    button2.caption := '&Delete'  ;
    button3.caption := '&Info'    ;
    button4.caption := '&Quit'    ;
    button1.onClick := groupAddUser ;
    button2.onClick := groupDelUser ;
    button3.onClick := objSeeObjectInfo ;
    button4.modalResult := mrCancel ;
    { Edit and Fill Up the listGrid }
    tempList := GetMemberList(inServer,
                              inObjectName,
                              True) ;
    listGrid.rowCount     := tempList.count+1 ;
    listGrid.cells[0,0]   := 'User ID'        ;
    listGrid.onDblClick   := button3.onClick  ;
    for nLoop := 1 to tempList.count do
      listGrid.cells[0,nloop] := tempList[nloop-1] ;
    if (listGrid.rowCount > 1) then
      begin
        listGrid.fixedRows := 1 ;
        listGrid.row := 1 ;
      end;
    screen.cursor := ncursor ;
    tempList.free ;
  end;


procedure TWinList.onPropertiesShow(sender : TObject)  ;
  { used when objEdit's 'Properties' button is selected }
  var
    ncursor  : TCursor ;
    tempList : TStringList ;
    nLoop    : word ;
  begin
    ncursor       := screen.cursor ;
    screen.cursor := crHourglass ;
    tempList      := TStringList.create ;
    caption       := inObjectName + ': Valid Properties' ;
    { Edit listGrid's Button Properties }
    { Loads onClick events, edits captions }
    button1.caption := '&Add'     ;
    button2.caption := '&Delete'  ;
    button3.caption := '&Info'    ;
    button4.caption := '&Quit'    ;
    button1.onClick := createNewProperty ;
    button2.onClick := deleteObjProperty ;
    button3.onClick := showPropertyInfo ;
    button4.modalResult := mrCancel ;
    { Edit and Fill Up the listGrid }
    listGrid.rowCount     := tempList.count+1 ;
    listGrid.cells[0,0]   := 'Property Name'     ;
    listGrid.onDblClick   := button3.onClick  ;
    tempList := getPropertyList(inServer,inObjectName,'*') ;
    listGrid.rowCount     := (tempList.count+1) ;
    for nLoop := 1 to tempList.count do
      listGrid.cells[0,nloop] := tempList[nloop-1] ;
    if (listGrid.rowCount > 1) then
      begin
        listGrid.fixedRows := 1 ;
        listGrid.row := 1 ;
      end;
    screen.cursor := ncursor ;
    tempList.free ;
  end;

procedure TWinList.onRightsShow(sender : TObject)  ;
  { used when objEdit's 'Rights' button is selected }
  var
    ncursor  : TCursor ;
    tempList : TStringList ;
    nLoop    : word ;
  begin
    ncursor       := screen.cursor ;
    screen.cursor := crHourglass ;
    tempList      := TStringList.create ;
    caption       := inObjectName + ': Trustee Directory Rights' ;
    { Edit listGrid's Button Properties }
    { Loads onClick events, edits captions }
    button1.caption := '&Add'     ;
    button2.caption := '&Delete'  ;
    button3.caption := '&Edit'    ;
    button4.caption := '&Quit'    ;
    button1.onClick := addNewRight ;
    button2.onClick := deleteRight ;
    button3.onClick := editRight ;
    button4.modalResult := mrCancel ;
    { Edit and Fill Up the listGrid }
    listGrid.rowCount     := tempList.count+1 ;
    listGrid.cells[0,0]   := 'Path/Rights'    ;
    listGrid.onDblClick   := button3.onClick  ;
    tempList := getTrusteeList(inServer,inObjectName) ;
    listGrid.rowCount     := (tempList.count+1) ;
    for nLoop := 1 to tempList.count do
      listGrid.cells[0,nloop] := tempList[nloop-1] ;
    if (listGrid.rowCount > 1) then
      begin
        listGrid.fixedRows := 1 ;
        listGrid.row := 1 ;
      end;
    tempList.free ;
    screen.cursor := ncursor ;
  end;

{ ******************* Stored Button Procedure Code ******************** }

procedure TWinList.editRight(sender : TObject) ;
  var
    rightsList : TNWRights   ;
    pathInfo   : TNWPathInfo ;
    tempList   : TStringList ;
    nloop      : word        ;
    cPath      : string ;
  begin
    cpath := strExtract(listGrid.cells[0,listGrid.row],' ',1) ;
    if parseNetwarePath(inServer,
                        cpath,
                        pathInfo) then 
      begin
        try
          application.createForm(TWinTrustee,winTrustee)           ;
          winTrustee.serverName.text := getServerName(inServer)    ;
          winTrustee.volName.text    := pathInfo.volumeName        ;
          winTrustee.pathName.text   := pathInfo.pathOnly          ;
          winTrustee.oldPath         := pathInfo.pathOnly          ;
          winTrustee.oldVolName      := pathInfo.volumeName        ;
          winTrustee.oldServer       := winTrustee.serverName.text ;
          winTrustee.nServer         := inServer     ;
          winTrustee.cUserID         := inObjectName ;
          winTrustee.creating        := false        ;
          {get rights, then enable rights checkboxes}
          if getObjectDirRights(inServer,
                                inObjectName,
                                cPath,
                                rightsList) then 
            begin
              with rightsList do begin
                winTrustee.lRead.checked          := read          ;
                winTrustee.lWrite.checked         := write         ;
                winTrustee.lCreate.checked        := create        ;
                winTrustee.lErase.checked         := erase         ;
                winTrustee.lModify.checked        := modify        ;
                winTrustee.lFileScan.checked      := fileScan      ;
                winTrustee.lAccessControl.checked := accessControl ;
                winTrustee.lSupervisor.checked    := supervisor    ;
              end;
              winTrustee.showModal ;
              if winTrustee.execute then
                okBox('Trustee Rights Edited;;Update Your Listbox Items Here');
            end;
        finally
          winTrustee.free ;
        end;
      end 
    else
      alertBox('Unknown Path Specification;;Play it Again, Sam') ; 
  end;

procedure TWinList.deleteRight(sender : TObject) ;
  var
    ctemp : string ;
  begin
    ctemp := listGrid.cells[0,listGrid.row] ;
    if noYesBox(ctemp +
                ';;Permanently Deleting Trustee Right;;Are You Sure?') then
      begin
        if deleteTrusteeRight(inServer,
                              inObjectName,
                              '',
                              ctemp) then
          okBox('Trustee Right Deleted;;Update Your Listbox Items Here')
        else
          alertBox('Error Deleting Trustee Right!') ;
      end;
  end;

procedure TWinList.addNewRight(sender : TObject) ;
  var
    rightsList : TNWRights ;
  begin
    try
      application.createForm(TWinTrustee,winTrustee) ;
      winTrustee.serverName.text := getServerName(inServer) ;
      winTrustee.volName.text    := 'SYS:' ;
      winTrustee.pathName.text   := '\'    ;
      winTrustee.nServer         := inServer     ;
      winTrustee.cUserID         := inObjectName ;
      winTrustee.creating        := true ;
      winTrustee.showModal ;
      if winTrustee.execute then
        okBox('Trustee Right Added;;Update Your Listbox Items Here');
    finally
      winTrustee.free ;
    end;
  end;

procedure TWinList.groupAddUser(sender : TObject) ;
  var
    cuser   : string ;
    tempList : TStringList ;
    nLoop    : word ;
  begin
    cuser := space(80) ;
    if inputQuery('Add User to Group ' + inObjectName,
                  'User Name',cUser) then
      begin
        try
          cuser := upperCase(allTrim(cuser)) ;
          if (getObjType(0,cuser) = nw_user) then
            begin
              tempList := TStringList.create  ;
              if addUserToGroup(inServer,
                                inObjectName,
                                cUser) then
                begin
                  { add new element to grid, then sort it }
                  tempList.add(cUser) ;
                  for nLoop := 1 to listGrid.rowCount do
                    tempList.add(listGrid.cells[nloop-1,0]) ;
                  tempList.sort ;
                  listGrid.fixedRows := 0 ;
                  listGrid.rowCount  := 1;
                  listGrid.rowCount  := tempList.count+1 ;
                  for nLoop := 1 to tempList.count do
                    listGrid.cells[nloop,0] := tempList[nloop-1];
                  if (listGrid.rowCount > 1) then
                    listGrid.fixedRows := 1 ;
                  okBox(cUser + ';Added to ' + inObjectName + ' Successfully') ;
                end
              else
                alertBox('Error!;Could Not Add ' + cUser + ' to ' + inObjectName + ';;Check Your Access Privileges') ;
            end
          else
            alertBox(cUser + ';;Is Not A User') ;
        finally
          tempList.free ;
        end;
      end;
  end;

procedure TWinList.userAddGroup(sender : TObject) ;
  { process 'Add member to Group' Button (specified in TWinObjEdit/Edit User) }
  var
    cgroup   : string ;
    tempList : TStringList ;
    nLoop    : word ;
  begin
    cgroup := space(80) ;
    if inputQuery('Add ' + inObjectName + ' to Group',
                  'Group Name',cgroup) then
      begin
        try
          cgroup := upperCase(allTrim(cgroup)) ;
          if (getObjType(0,cgroup) = nw_group) then
            begin
              tempList := TStringList.create  ;
              if addUserToGroup(inServer,
                                cgroup,
                                inObjectName) then
                begin
                  { add new element to grid, then sort it }
                  tempList.add(inObjectName) ;
                  for nLoop := 1 to listGrid.rowCount do
                    tempList.add(listGrid.cells[nloop-1,0]) ;
                  tempList.sort ;
                  listGrid.fixedRows := 0 ;
                  listGrid.rowCount  := 1;
                  listGrid.rowCount  := tempList.count+1 ;
                  for nLoop := 1 to tempList.count do
                    listGrid.cells[nloop,0] := tempList[nloop-1];
                  if (listGrid.rowCount > 1) then
                    listGrid.fixedRows := 1 ;
                  okBox(inObjectName + ';Added to Group Successfully') ;
                end
              else
                alertBox('Error!;Could Not Add ' + inObjectName + ' to ' + cgroup + ';;Check Your Access Privileges') ;
            end
          else
            alertBox(cGroup + ';;Is Not A Group') ;
        finally
          tempList.free ;
        end;
      end;
  end;

procedure TWinList.userDelFromGroup(sender : TObject) ;
  { process 'Delete Group From User'  Button (specified in TWinObjEdit) }
  var
    ctemp : string ;
  begin
    ctemp := listGrid.cells[0,listGrid.row] ;
    if YesNoBox(inObjectName + ';Deleting User from Group ' +
                ctemp + ';;Are You Sure?') then
      begin
        if deleteUserFromGroup(inServer,
                               ctemp,
                               inObjectName) then
          begin
            listGrid.cells[0,listGrid.row] := '<Deleted>' ;
            okBox(ctemp + ';;Removed from ' + inObjectName) ;
          end
        else
          alertBox('Error Deleting User from Group;;Check Your Access Rights') ;
      end;
  end;

procedure TWinList.groupDelUser(sender : TObject) ;
  { process 'Delete User from Group' Button }
  var
    ctemp : string ;
  begin
    ctemp := listGrid.cells[0,listGrid.row] ;
    if YesNoBox('Removing User ' + ctemp + ';From Group ' +
                inObjectName + ';;Are You Sure?') then
      begin
        if deleteUserFromGroup(inServer,
                               inObjectName,
                               ctemp) then
          begin
            listGrid.cells[0,listGrid.row] := '<Deleted>' ;
            okBox(ctemp + ';;Removed from ' + inObjectName) ;
          end
        else
          alertBox('Error Deleting User from Group;;Check Your Access Rights') ;
      end;
  end;
  

procedure TWinList.objSeeObjectInfo(sender : TObject) ;
  { process Group's 'Info' Button }
  var
    ctemp : string ;
  begin
    ctemp := listGrid.cells[0,listGrid.row] ;
    okBox('Object Name:  ' + ctemp + ';Full Name:  ' + fullName(0,ctemp)) ;
  end;

procedure TWinList.showPropertyInfo(sender : TObject) ;
  var
    ctemp : string ;
  begin
    ctemp := listGrid.cells[0,listGrid.row] ;
    OKBox(ctemp) ;  
  end;

procedure TWinList.createNewProperty(sender: TObject) ;
  var
    readSecurity,
    writeSecurity,
    objTypeFlag    : TNWFlags ;
    permanent      : boolean ;
    ctext          : string ;
  begin
    ctext         := space(80) ;
    objTypeFlag   := BF_ITEM ;
    permanent     := true ;
    readSecurity  := BS_LOGGED_READ ;
    writeSecurity := BS_OBJECT_WRITE ;
    if inputQuery(inObjectName + ': Creating New Property','Property Name:',ctext) then
      begin
        if createProperty(
                          inServer,
                          inObjectName,
                          getObjType(0,inObjectName),
                          ctext,
                          objTypeFlag,
                          permanent,
                          readSecurity,
                          writeSecurity) then
          okBox(ctext + ';;Property Created Successfully')
        else
          alertBox(ctext + ';Error Creating Property') ;
      end;
  end;

procedure TWinList.deleteObjProperty(sender : TObject) ;
  var
    ctemp : string ;
  begin
    ctemp := strExtract(listGrid.cells[0,listGrid.row],' ',1) ;
    if noYesBox(ctemp + ';Permanently Removing Property;;Are You Sure?') then
      begin
        if deleteProperty(inServer,inObjectName,ctemp) then
          okBox(ctemp + ';;Property Deleted')
        else
          alertBox(ctemp + ';;Error Deleting Property!') ;
      end;
  end;


procedure TWinList.queueJobDelete(sender : TObject) ;
  var
    jobInfo : TNWQueueJobInfo ;
  begin
    if noYesBox('Removing Job From Queue;;Are You Sure') then
      begin
        with jobInfo do begin
          nServer         := inServer ;
          cQueue          := getObjName(inServer,inQueue) ;
          jobID           := strToIntDef(listGrid.cells[1,listGrid.row],0)  ;
        end;
        if (not deleteQueueJob(jobInfo)) then
          alertBox('Error Removing Job from Queue') ;
        onPrintJobsShow(sender) {refresh list}
      end;
  end;
    
procedure TWinList.queueJobInfo(sender : TObject) ;
  var
    jobInfo : TNWQueueJobInfo ;
  begin
    with jobInfo do begin
      nServer := inServer ;
      cQueue  := getObjName(inServer,inQueue) ;
      jobID   := strToIntDef(listGrid.cells[1,listGrid.row],0)  ;
    end;
    if getQueueJobInfo(jobInfo) then
      okBox('Job Info;;' + 
            'Owner: '    + jobInfo.ownerName + ';' + 
            'Server: '   + jobInfo.serverName + ';' + 
            'QServer: '  + jobInfo.queueServerName + ';' + 
            'Text: '     + jobInfo.jobDescription + ';' + 
            'FileName: ' + jobInfo.jobFileName + ';' + 
            'EntryDate: ' + dateTimeToStr(jobInfo.entryDateTime) + ';' +
            'ExecDate: ' + dateTimeToStr(jobInfo.execDateTime) )
    else
      begin
        alertBox('Could Not Read Job Information') ;
        onPrintJobsShow(sender) ;
      end;
  end;

{ End of Unit }
end.
